<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Brand;

class BrandController extends Controller
{
    /* =========================
       CREATE PAGE
    ========================= */
    public function create()
    {
        return view('brand.create');
    }

    /* =========================
       STORE (CREATE BRAND)
       PDF MAX: 1 MB
    ========================= */
    public function store(Request $r)
    {
        // ✅ VALIDATION (1 MB PDF)
        $r->validate([
            'name'              => 'required|string|max:255',
            'website'           => 'required|url',
            'brand_rm_name'     => 'required|string',
            'sector'            => 'required|string',
            'brand_storetype'   => 'required|string',
            'business_model'    => 'required|array',
            'deck'              => 'required|mimes:pdf|max:1024', // 🔥 1 MB
        ]);

        $data = $r->all();

        // ✅ BUSINESS MODEL ARRAY → STRING
        if ($r->has('business_model')) {
            $data['business_model'] = implode(',', $r->business_model);
        }

        // ✅ STORE PDF
        if ($r->hasFile('deck')) {
            $data['deck'] = $r->file('deck')->store('brand_decks', 'public');
        }

        Brand::create($data);

        return back()->with('success', 'Brand Added');
    }

    /* =========================
       LIST
    ========================= */
    public function list(Request $request)
    {
        return Brand::orderBy('id', 'desc')->paginate(23);
    }

    /* =========================
       SHOW
    ========================= */
    public function show($id)
    {
        return Brand::findOrFail($id);
    }

    /* =========================
       UPDATE (EDIT BRAND)
       PDF MAX: 1 MB
    ========================= */
    public function update(Request $request, $id)
    {
        $brand = Brand::findOrFail($id);

        // ✅ VALIDATION (PDF OPTIONAL, MAX 1 MB)
        $request->validate([
            'name'              => 'required|string|max:255',
            'website'           => 'nullable|url',
            'brand_rm_name'     => 'required|string',
            'sector'            => 'required|string',
            'brand_storetype'   => 'required|string',
            'business_model'    => 'required|array',
            'deck'              => 'nullable|mimes:pdf|max:1024', // 🔥 1 MB
        ]);

        $data = $request->except(['_token', '_method']);

        // ✅ BUSINESS MODEL ARRAY → STRING
        if ($request->has('business_model')) {
            $data['business_model'] = implode(',', $request->business_model);
        }

        // ✅ UPDATE PDF (IF UPLOADED)
        if ($request->hasFile('deck')) {
            $data['deck'] = $request->file('deck')->store('brand_decks', 'public');
        }

        $brand->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Brand updated successfully'
        ]);
    }

    /* =========================
       DELETE
    ========================= */
    public function destroy($id)
    {
        Brand::findOrFail($id)->delete();

        return response()->json([
            'success' => true
        ]);
    }
}
