<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\State;
use App\Models\City;
use App\Models\Market;
use Illuminate\Validation\Rule;

class LocationController extends Controller
{
    /* ===================== FETCH ===================== */

    public function states(){
        return State::orderBy('name')->get();
    }

    public function cities($state_id){
        return City::where('state_id',$state_id)->get();
    }

    public function markets($city_id){
        return Market::where('city_id',$city_id)->get();
    }

    /* ===================== STORE ===================== */

    // ✅ STORE STATE (unique globally)
    public function storeState(Request $r){
        $r->validate([
            'name' => 'required|string|unique:states,name'
        ]);

        return State::create([
            'name' => $r->name
        ]);
    }

    // ✅ STORE CITY (unique per state)
    public function storeCity(Request $r){
        $r->validate([
            'state_id' => 'required|exists:states,id',
            'name' => [
                'required',
                Rule::unique('cities')->where(function ($q) use ($r) {
                    return $q->where('state_id', $r->state_id);
                })
            ]
        ]);

        return City::create([
            'state_id' => $r->state_id,
            'name'     => $r->name
        ]);
    }

    
   // ✅ STORE MARKET (unique per city)
public function storeMarket(Request $r)
{
    $r->validate([
        'city_id' => 'required|exists:cities,id',
        'name' => [
            'required',
            'string',
            Rule::unique('markets')->where(function ($q) use ($r) {
                return $q->where('city_id', $r->city_id);
            })
        ]
    ]);

    return Market::create([
        'city_id' => $r->city_id,
        'name'    => $r->name
    ]);
}


    /* ===================== PAGINATION ===================== */

    // PAGINATED STATES
    public function statesPaginated(){
        return State::orderBy('id','desc')->paginate(5);
    }

    // PAGINATED CITIES
    public function citiesPaginated($state_id){
        return City::where('state_id', $state_id)
            ->orderBy('id','desc')
            ->paginate(5);
    }

    // PAGINATED MARKETS
    public function marketsPaginated($city_id){
        return Market::where('city_id', $city_id)
            ->orderBy('id','desc')
            ->paginate(5);
    }




    // UPDATE STATE
public function updateState(Request $r, $id)
{
    $r->validate([
        'name' => 'required|string|unique:states,name,'.$id
    ]);

    $state = State::findOrFail($id);
    $state->update(['name'=>$r->name]);

    return response()->json(['success'=>true]);
}

// UPDATE CITY
public function updateCity(Request $r, $id)
{
    $city = City::findOrFail($id);

    $r->validate([
        'name' => [
            'required',
            Rule::unique('cities')->where(fn($q)=>
                $q->where('state_id',$city->state_id)
            )->ignore($id)
        ]
    ]);

    $city->update(['name'=>$r->name]);

    return response()->json(['success'=>true]);
}

// UPDATE MARKET
public function updateMarket(Request $r, $id)
{
    $market = Market::findOrFail($id);

    $r->validate([
        'name' => [
            'required',
            Rule::unique('markets')->where(fn($q)=>
                $q->where('city_id',$market->city_id)
            )->ignore($id)
        ]
    ]);

    $market->update(['name'=>$r->name]);

    return response()->json(['success'=>true]);
}

}